/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  Param.h - Paramater/Parameters group definition class
 *
 *  Version: $Id: Param.h 2752 2006-01-19 14:40:17Z mst $
 *
 */

#ifndef PARAM_H
#define PARAM_H

#include <stdlib.h>

#include <string>
#include <map>
#include <list>

#include "Attributes.h"
#include "compatibility.h"
#include "utils.h"
#include "MultiFile.h"

//------------------------------------------
//------------------------------ Param class
//------------------------------------------
//

class MultiFile;
class MSection;
class Enum;
class Group;
class Param : public ErrMsg
{
public:
    enum Repr { UNS, INT, UNS64, INT64, ENUM, ASCII, BOOL};
    enum Type { C, CRSPACE, DUMMY};
    enum      { GEO_DEF = 0xffffffff };  // Geo address default

    Param(const char *n) : name(n), boot2(false)         {}
    Param(const std::string& n) : name(n), boot2(false)  {}
    ~Param()                              { }

    static bool            warn;      // print warnings
    static DeviceType      dev_type;  // dev_type effects the get() function return string format. 

    bool                   allow_geo; // This parameter may have GEO id

    // Convert value to required type, check it (if necessary)
    // and assign to paramer
    bool                   assign(const std::string& sval,
                                  const u_int32_t geo = GEO_DEF)
                           { return assign(sval, "", -1, geo); }
    bool                   assign(const std::string& sval,
                                  const std::string& w_f, 
                                  const int w_l,
                                  const u_int32_t geo = GEO_DEF, 
                                  const bool ext_assignment = false);
    std::string            get(const   u_int32_t geo = GEO_DEF);
    std::string            geti(const  u_int32_t geo = GEO_DEF);
    u_int32_t              get32(const u_int32_t geo = GEO_DEF);
    u_int64_t              get64(const u_int32_t geo = GEO_DEF);

    std::string            GetAssignmentDescription(const std::string& line_prefix);

    std::string            name;     // Parameter name  
    std::string            refname;  // Parameter refname, backward comp. for tavor.
    std::string            fname;    // File name where parameter is defined
    int                    line;     // Line number where parameters' def. starting
    Group                  *group;   // Group where parameter is defined

    Type                   type;     // Type of parameter (C/CRSPACE/DUMMY)
    Repr                   repr;     // How to represent the variable
    bool                   hidden;   // Is parameter is hidden
    bool                   ro;       // Is parameter read only
    std::string            descr;    // Description

    std::string            def;      // Default value string

    // Parameter value
    struct Value {
        u_int32_t   i;  // Type == INT/UNS/ENUM
        u_int64_t   l;  // Type == INT64/UNS64
        std::string s;  // Type == ENUM/ASCII
        std::string where_f;  // File name where parameter get its value
        int         where_l;  // Line number where parameter get its value
        bool        ext_val;  // Assigned from a configuration file or command line
    };
    Attributes<u_int32_t, Value> values;  // Parameter values per GEO id

    // Follow relevant iff Repr == INT or UNS
    bool                   check_r; // Check range
    u_int32_t              min;     // Minimal value (relevant iff check_r==true)
    u_int32_t              max;     // Maximal value (relevant iff check_r==true)

    // Follow relevant iff Repr == INT64 or UNS64
    //bool                   check_r;// Same as Repr == INT or UNS
    u_int64_t              lmin;     // Minimal value (relevant iff check_r==true)
    u_int64_t              lmax;     // Maximal value (relevant iff check_r==true)


    // Follow relevant iff Repr == ENUM
    Enum                   *enump;

    // Follow relevant iff Repr == ASCII
    u_int32_t              slength;  // Maximal length of string variable

    // Follow relevant iff Type == C
    u_int32_t              offset;   // Offset in C code in bytes
    MSection               *sect;    // Section contains relevant C-code

    // follow relevant iff Type == CRSPACE
    //u_int32_t              offset; // Offset in CRSpace in bytes.
                                     // Same as if Type == C
    u_int32_t              bitoffs;  // Offset of field inside word
    u_int32_t              bitsize;  // Size of field inside word
    bool                   boot2;    // True if reg is initialized in boot2

    // Backward compatibility to infiniburn. 
    // The expressions are to be moved to the perl section, and not to be used
    // in the future.
    std::string            expr;

    // follow relevant iff this is special parameter
    // otherwise tag_id == 0;
    int                    tag_id;   // Special TAG ID
    u_int32_t              tag_offs; // Offset inside special struct

};

//------------------------------------------
//------------------------------ Group class
//------------------------------------------
//
class Group
{
public:
    Group(const char *n) : name(n)         { }
    Group(const std::string& n) : name(n)  { }
    ~Group()                               { }

    std::string                   name;    // Group name
    std::string                   fname;   // File name where group is defined
    std::string                   descr;    // Description
    int                           line;    // Line number where group def. starting
    std::map<std::string, Param*> prmap;   // Map of all parameters of this group
    std::list<Param*>             prlist;  // List of such parametes in an orig. order

    std::map<std::string, u_int32_t> dup_params; // Counts number of params with same name in the group. 
                                                 // These params are auto renamed with the curr index (patch for tavor)
};

//-----------------------------------------
//------------------------------ Enum class
//-----------------------------------------
//
class Enum : public ErrMsg
{
public:
    Enum(const char *n) : name(n)        { }
    Enum(const std::string& n) : name(n) { }
    ~Enum()                              { }

    bool append(const std::string& enum_name, const u_int32_t enum_val);

    std::string                      name;   // Enum name
    std::string                      fname;  // File name where group is defined
    int                              line;   // Line number where group def. starting
    std::map<std::string, u_int32_t> vals;   // Map of keys/values
    std::list<std::string>           keys;   // List of keys (to keep orig. order)
};

#endif
